import { fileURLToPath } from 'node:url'
import VueI18nPlugin from '@intlify/unplugin-vue-i18n/vite'
import vue from '@vitejs/plugin-vue'
import vueJsx from '@vitejs/plugin-vue-jsx'
import AutoImport from 'unplugin-auto-import/vite'
import Components from 'unplugin-vue-components/vite'
import { VueRouterAutoImports, getPascalCaseRouteName } from 'unplugin-vue-router'
import VueRouter from 'unplugin-vue-router/vite'
import { defineConfig } from 'vite'
import VueDevTools from 'vite-plugin-vue-devtools'
import Layouts from 'vite-plugin-vue-layouts'
import vuetify from 'vite-plugin-vuetify'
import dotenv from 'dotenv'
import { VitePWA } from 'vite-plugin-pwa'
import { dirname } from 'path';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

console.log('filename', __filename, 'dirname', __dirname)

dotenv.config({ path: `${__dirname}/../.env.prod` });
const isProduction = true

const timestamp = new Date().getTime()

// https://vitejs.dev/config/
export default defineConfig({
  plugins: [
    VitePWA({
      registerType: 'autoUpdate',
      base: './',
      strategies: 'injectManifest',
      srcDir: '/var/www/public/client/client-user/apk',
      injectManifest: {
        rollupFormat: 'life',
        maximumFileSizeToCacheInBytes: 3000000
      },
      manifest: {
        name: 'Akkiperto',
        short_name: 'Akkiperto',
        theme_color: '#F27405',
        icons: [
          {
            src: 'https://app.akkiperto.com.br/uploads/assets-public/pwa-192.png',
            sizes: '192x192',
            type: 'image/png',
          },
          {
            src: 'https://app.akkiperto.com.br/uploads/assets-public/pwa-512.png',
            sizes: '512x512',
            type: 'image/png',
          },
        ],
      }
    }),
    // Docs: https://github.com/posva/unplugin-vue-router
    // ℹ️ This plugin should be placed before vue plugin
    VueRouter({
      getRouteName: routeNode => {
        // Convert pascal case to kebab case
        return getPascalCaseRouteName(routeNode)
          .replace(/([a-z0-9])([A-Z])/g, '$1-$2')
          .toLowerCase()
      },
    }),
    vue({
      template: {
        compilerOptions: {
          isCustomElement: tag => tag === 'swiper-container' || tag === 'swiper-slide',
        },
      },
    }),
    VueDevTools(),
    vueJsx(),

    // Docs: https://github.com/vuetifyjs/vuetify-loader/tree/master/packages/vite-plugin
    vuetify({
      styles: {
        configFile: 'src/assets/styles/variables/_vuetify.scss',
      },
    }),

    // Docs: https://github.com/johncampionjr/vite-plugin-vue-layouts#vite-plugin-vue-layouts
    Layouts({
      layoutsDirs: './src/layouts/',
    }),

    // Docs: https://github.com/antfu/unplugin-vue-components#unplugin-vue-components
    Components({
      dirs: ['src/@core/components', 'src/views/demos', 'src/components'],
      dts: true,
      resolvers: [
        componentName => {
          // Auto import `VueApexCharts`
          if (componentName === 'VueApexCharts')
            return { name: 'default', from: 'vue3-apexcharts', as: 'VueApexCharts' }
        },
      ],
    }),

    // Docs: https://github.com/antfu/unplugin-auto-import#unplugin-auto-import
    AutoImport({
      imports: ['vue', VueRouterAutoImports, '@vueuse/core', '@vueuse/math', 'vue-i18n', 'pinia'],
      dirs: [
        '../@global/*',
        './src/@core/utils',
        './src/@core/composable/',
        './src/composables/',
        './src/utils/',
        './src/plugins/*/composables/*',
      ],
      vueTemplate: true,

      // ℹ️ Disabled to avoid confusion & accidental usage
      ignore: ['useCookies', 'useStorage'],
      eslintrc: {
        enabled: true,
        filepath: './.eslintrc-auto-import.json',
      },
    }),

    // Docs: https://github.com/intlify/bundle-tools/tree/main/packages/unplugin-vue-i18n#intlifyunplugin-vue-i18n
    VueI18nPlugin({
      runtimeOnly: true,
      compositionOnly: true,
      include: [
        fileURLToPath(new URL('./src/plugins/i18n/locales/**', import.meta.url)),
      ],
    })
  ],
  server: {
    host: '0.0.0.0',
    https: isProduction ? {
      key: '/etc/letsencrypt/live/akkiperto.com.br/privkey.pem',
      cert: '/etc/letsencrypt/live/akkiperto.com.br/fullchain.pem'
    } : false,
    proxy: {
      '/api': {
        target: process.env.VITE_API_BASE_URL, // `https://app.akkiperto.com.br:3000
        changeOrigin: true,
        rewrite: path => path.replace(/^\/api/, ''),
      },
      '/uploads': {
        target: process.env.VITE_API_BASE_URL, // `https://app.akkiperto.com.br:3000
        changeOrigin: true,
      },
      '/download': {
        target: process.env.VITE_API_BASE_URL, // `https://app.akkiperto.com.br:3000
        changeOrigin: true,
      },
      '/purchase': {
        target: process.env.VITE_API_BASE_URL, // `https://app.akkiperto.com.br:3000
        changeOrigin: true,
      },
      '/.well-known': {
        target: process.env.VITE_API_BASE_URL, // `https://app.akkiperto.com.br:3000
        changeOrigin: true,
      },
    },
  },
  define: {
    'process.env': {
      VITE_API_BASE_URL: process.env.VITE_API_BASE_URL,
      VITE_KEY_MAPS: process.env.VITE_KEY_MAPS,
      VITE_ZAP_BASE_URL: process.env.VITE_ZAP_BASE_URL,
      VITE_VAPP: process.env.VITE_VAPP,
    }
  },
  resolve: {
    alias: {
      '@': fileURLToPath(new URL('./src', import.meta.url)),
      '@themeConfig': fileURLToPath(new URL('./themeConfig.js', import.meta.url)),
      '@core': fileURLToPath(new URL('./src/@core', import.meta.url)),
      '@layouts': fileURLToPath(new URL('./src/@layouts', import.meta.url)),
      '@images': fileURLToPath(new URL('./src/assets/images/', import.meta.url)),
      '@fotoperfil': fileURLToPath(new URL('./src/assets/images/fotos-perfil/', import.meta.url)),
      '@styles': fileURLToPath(new URL('./src/assets/styles/', import.meta.url)),
      '@configured-variables': fileURLToPath(new URL('./src/assets/styles/variables/_template.scss', import.meta.url)),
      'apexcharts': fileURLToPath(new URL('node_modules/apexcharts-clevision', import.meta.url)),
      '@db': fileURLToPath(new URL('./src/plugins/fake-api/handlers/', import.meta.url)),
      '@api-utils': fileURLToPath(new URL('./src/plugins/fake-api/utils/', import.meta.url)),
      '@global': fileURLToPath(new URL('../@global', import.meta.url)),
      '@vuetify': fileURLToPath(new URL('./node_modules/vuetify', import.meta.url)),
      'vuetify/lib/components': fileURLToPath(new URL('./node_modules/vuetify/lib/components', import.meta.url)),
      'vuetify/labs': fileURLToPath(new URL('./node_modules/vuetify/lib/labs', import.meta.url)),
      '@vueuse/core': fileURLToPath(new URL('./node_modules/@vueuse/core', import.meta.url)),
    },
  },
  build: {
    chunkSizeWarningLimit: 5000,
    rollupOptions: {
      output: {
        entryFileNames: `assets/[name].[hash].${timestamp}.js`,
        chunkFileNames: `assets/[name].[hash].${timestamp}.js`,
        assetFileNames: `assets/[name].[hash].${timestamp}.[ext]`,
      },
    },
  },
  optimizeDeps: {
    exclude: ['vuetify'],
    entries: [
      './src/**/*.vue',
    ],
  },
})